<?php
/**
 *  @copyright 2017  WPVKP  https://wpvkp.com
 *
 *  This plugin is inspired from WP Speed of Light by JoomUnited.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
defined('ABSPATH') || die('No direct script access allowed!');

class Ultimateoptimizer_Admin {
    public function __construct(){
        add_action('init', function(){
            load_plugin_textdomain('ultimateoptimizer', false, dirname(plugin_basename(__FILE__)) . '/languages/');
        });

        // Add our custom action to clear cache
        add_action('ultimateoptimizer_clear_all_cache', array($this, 'ultimateoptimizer_clear_all_cache'));
        add_action('ultimateoptimizer_clear_varnish', array($this, 'ultimateoptimizer_clear_varnish'));

        add_action('admin_init', array($this, 'admin_init'));
        //register menu
        add_action('admin_menu', array($this, 'register_menu_page'));
        add_action('network_admin_menu', array($this, 'register_network_menu_page'));

        // Add notice when installing plugin
	    $first_install = get_option('ultimateoptimizer_first_install');
	    if ($first_install === false) {
	    	add_option('ultimateoptimizer_first_install', 'yes');
	    }
	    if ($first_install == 'yes') {
		    add_action('admin_notices', array($this, 'installing_notices'));
	    }

        $config = get_option('ultimateoptimizer_basic_settings');

        if(isset($config['ultimateoptimizer-display-clean']) && $config['ultimateoptimizer-display-clean']){
            //register top bar menu
            add_action('admin_bar_menu', array($this, 'register_admin_bar_menu'), 999);
        }

        /** Load admin js * */
        add_action('admin_enqueue_scripts', array($this, 'loadAdminScripts'));

        add_action('wp_head', array($this,'define_ajaxurl'));
        $this->ajaxHandle();

        // Add setting buttons to plugins list page
	    add_filter('plugin_action_links_' . ULTIMATEOPTIMIZER_BASENAME, array($this, 'ultimateoptimizer_add_action_links'));
        add_filter('network_admin_plugin_action_links_' . ULTIMATEOPTIMIZER_BASENAME, array($this, 'ultimateoptimizer_add_action_links_network'));
    }

    /**
     * Admin Init
     *
     */
    public function admin_init()
    {
        //Check plugin requirements
        if (version_compare(PHP_VERSION, '5.3', '<')) {
            if (current_user_can('activate_plugins') && is_plugin_active(plugin_basename(__FILE__))) {
                deactivate_plugins(__FILE__);
                add_action('admin_notices', array($this, 'ultimateoptimizer_show_error'));
                unset($_GET['activate']);
            }
        }
        //Do not load anything more
        return;
    }

    //define ajaxurl
    function define_ajaxurl() {
        if(current_user_can('manage_options')){
            echo '<script type="text/javascript">
           var ajaxurl = "' . admin_url('admin-ajax.php') . '";
             </script>';
        }
    }

    // Add notice message when install plugin
	public function installing_notices() {
		$class = 'notice notice-success';
		$message = __('Thanks for installing Ultimateoptimizer. It is always recommended not to use more than one caching plugin at the same time. We recommend you to purge cache if necessary.', 'ultimateoptimizer');

		printf( '<div class="%1$s"><p>%2$s <button class="button" id="ultimateoptimizer-hide-install-msg">'.__("Hide message", "ultimateoptimizer").'</button></p></div>', esc_attr( $class ), esc_html( $message ));
		update_option('ultimateoptimizer_first_install', 'no');
	}


	function loadAdminScripts() {
        wp_enqueue_script('jquery');
        wp_enqueue_script('ultimateoptimizer-backend', plugins_url('assets/js/ultimateoptimizer-backend.js', dirname(__FILE__)), array('jquery'), ULTIMATEOPTIMIZER_VERSION, true);
        wp_enqueue_style('ultimateoptimizer-topbar', plugins_url('assets/css/topbar.css', dirname(__FILE__)));
        $current_screen = get_current_screen();
        if($current_screen->base == 'settings_page_ultimateoptimizer' || $current_screen->base == 'settings_page_ultimateoptimizer-network'){
            //add css
            wp_enqueue_style('ultimateoptimizer-style', plugins_url('assets/css/style.css', dirname(__FILE__)));
            //js
            wp_enqueue_script('ultimateoptimizer-configuration', plugins_url('assets/js/ultimateoptimizer-configuration.js', dirname(__FILE__)), array('jquery'), ULTIMATEOPTIMIZER_VERSION, true);
        }

        $token_name = array(
            'ultimateoptimizer_purge_varnish' => wp_create_nonce("_ultimateoptimizer_purge_varnish"),
            'ultimateoptimizer_purge_database' => wp_create_nonce("_ultimateoptimizer_purge_database"),
            'ultimateoptimizer_purge_cache' => wp_create_nonce("_ultimateoptimizer_purge_cache"),
            'purge_all_href' => wp_nonce_url(add_query_arg('ultimateoptimizer_purge', 1), 'ultimateoptimizer_purge_cache'),
        );

        wp_localize_script('ultimateoptimizer-backend','ultimateoptimizer_token_name',$token_name);
    }

    /**
     * Register menu
     *
     */
    function register_menu_page()
    {
        //add submenu for cloudsway
        add_submenu_page( 'options-general.php',  __('Ultimate Optimizer', 'ultimateoptimizer'),   __('Ultimate Optimizer', 'ultimateoptimizer'),  'manage_options',  'ultimateoptimizer', array($this, 'ultimateoptimizer_load_page')  );
    }

    function register_network_menu_page()
    {
    	//add submenu for multisite network
    	add_submenu_page('settings.php',  __('Ultimate Optimizer', 'ultimateoptimizer'),   __('Ultimate Optimizer', 'ultimateoptimizer'),  'manage_options',  'ultimateoptimizer', array($this, 'ultimateoptimizer_load_page'));
    }


    /**
     * Register bar menu
     *
     */
    function register_admin_bar_menu(WP_Admin_Bar $wp_admin_bar)
    {
        if (current_user_can('manage_options') || current_user_can('editor')) {
            // add a parent item
            $args = array(
                'id' => 'ultimateoptimizer-topbar',
                'title' => esc_html(__('Ultimate Optimizer', 'ultimateoptimizer')),
                'meta' => array(
                    'classname' => 'ultimateoptimizer',
                ),
            );
            $wp_admin_bar->add_node( $args );

            // add purge all item
            $args = array(
                'id'     => 'ultimateoptimizer-purge-all',
                'title'  => esc_html(__('Purge All Cache', 'ultimateoptimizer')),
                'parent' => 'ultimateoptimizer-topbar',
                'meta'   => array( 'class' => 'ultimateoptimizer-toolbar-group' ),
            );
            $wp_admin_bar->add_node( $args );

            // Editor role can only use Purge all cache option
            if (current_user_can('editor')) return $wp_admin_bar;

            // add purge modules group
            $args = array(
                'id'     => 'ultimateoptimizer-purge-modules',
                'title'  => esc_html(__('Purge Modules', 'ultimateoptimizer')),
                'parent' => 'ultimateoptimizer-topbar',
                'meta'   => array( 'class' => 'ultimateoptimizer-toolbar-group' ),
            );
            $wp_admin_bar->add_node( $args );


            // add child item (Purge Modules)
            $args = array(
                'id'     => 'ultimateoptimizer-purge-varnish-group',
                'title'  => esc_html(__('Purge Varnish Cache', 'ultimateoptimizer')),
                'parent' => 'ultimateoptimizer-purge-modules',
            );
            $wp_admin_bar->add_node( $args );

            // add child item (Purge Modules)
            $args = array(
                'id'     => 'ultimateoptimizer-purge-file-group',
                'title'  => esc_html(__('Purge Internal Cache', 'ultimateoptimizer')),
                'parent' => 'ultimateoptimizer-purge-modules',
            );
            $wp_admin_bar->add_node( $args );

            // add settings item
            $args = array(
                'id'     => 'ultimateoptimizer-settings',
                'title'  => esc_html(__('Settings', 'ultimateoptimizer')),
                'parent' => 'ultimateoptimizer-topbar',
                'href' => 'options-general.php?page=ultimateoptimizer',
                'meta'   => array( 'class' => 'ultimateoptimizer-toolbar-group' ),
            );
            $network_settings_url = network_admin_url('settings.php?page=ultimateoptimizer');
            if (is_multisite())
            	$args = array(
		            'id'     => 'ultimateoptimizer-settings',
		            'title'  => esc_html(__('Settings', 'ultimateoptimizer')),
		            'parent' => 'ultimateoptimizer-topbar',
		            'href' => $network_settings_url,
		            'meta'   => array( 'class' => 'ultimateoptimizer-toolbar-group' ),
	            );
            $wp_admin_bar->add_node( $args );

            // add support item
            $args = array(
                'id'     => 'ultimateoptimizer-support',
                'title'  => esc_html(__('Support', 'ultimateoptimizer')),
                'href' => 'https://wpvkp.com/support/',
                'parent' => 'ultimateoptimizer-topbar',
                'meta'   => array( 'class' => 'ultimateoptimizer-toolbar-group',
                    'target' => '_blank'),
            );
            $wp_admin_bar->add_node( $args );
        }
    }

    function ultimateoptimizer_load_page()
    {
        if (isset($_GET['page']) && $_GET['page'] == 'ultimateoptimizer') {
            require_once (ULTIMATEOPTIMIZER_PLUGIN_DIR . 'views/ultimateoptimizer-setting-views.php');
        }
    }

    public function ultimateoptimizer_show_error()
    {
        echo '<div class="error"><p><strong>Ultimateoptimizer</strong> need at least PHP 5.3 version, please update php before installing the plugin.</p></div>';
    }
    //ajax admin
    function ajaxHandle() {
        add_action('wp_ajax_ultimateoptimizer_purge_varnish', array('Ultimateoptimizer_Configuration', 'purge_varnish_action'));
        add_action('wp_ajax_ultimateoptimizer_purge_file', array('Ultimateoptimizer_Configuration', 'ultimateoptimizer_ajax_clean_cache'));
        add_action('wp_ajax_ultimateoptimizer_purge_database', array('Ultimateoptimizer_Configuration', 'ultimateoptimizer_ajax_purge_database'));
    }
    /*
     * Register active plugin hook
     */
    public static function plugin_active_hook(){
        WP_Filesystem();
        // Default basic
        $basic = get_option('ultimateoptimizer_basic_settings');
        if(empty($basic)) $basic = array();
        $default_basic = array(
            'ultimateoptimizer-active' => '1',
            'ultimateoptimizer-ttl' => '',
            'ultimateoptimizer-minify-html' => '0',
            'ultimateoptimizer-minify-css' => '0',
            'ultimateoptimizer-minify-js' => '0',
            'ultimateoptimizer-desktop-cache' => '1',
            'ultimateoptimizer-mobile-cache' => '1',
            'ultimateoptimizer-disable-admin' => '1',
            'ultimateoptimizer-display-clean' => '1',
            'ultimateoptimizer-include-inline-js' => '0',
            'ultimateoptimizer-include-inline-css' => '0',
        );
        $basic= array_merge($default_basic,$basic);




        // Default must
        $must = get_option('ultimateoptimizer_must_settings');
        if(empty($must)) $must = array();
        $default_must = array(
            'ultimateoptimizer-remove-query-strings'     => '0',
            'ultimateoptimizer-remove-wp-embed-js'       => '0',
            'ultimateoptimizer-remove-wp-emoji'          => '0',
            'ultimateoptimizer-remove-wp-rsd'            => '0',
            'ultimateoptimizer-remove-wp-gen'            => '0',
            'ultimateoptimizer-remove-wp-feeds'          => '0',
            'ultimateoptimizer-remove-mfest'             => '0',
            'ultimateoptimizer-remove-wp-shortlink'      => '0',
            'ultimateoptimizer-enable-gzip'              => '0',
            'ultimateoptimizer-enable-browser-cache'     => '0',
            'ultimateoptimizer-unset-etag'               => '0',
        );
        $must= array_merge($default_must,$must);




        // Default Advanced
        $advanced = get_option('ultimateoptimizer_advanced_settings');
        if(empty($advanced)) $advanced = array();
        $default_advanced = array(
            'ultimateoptimizer-exclude-urls' => array(),
            'ultimateoptimizer-group-css' => '0',
            'ultimateoptimizer-group-js' => '0',
            'ultimateoptimizer-exclude-css' => array(),
            'ultimateoptimizer-exclude-js' => array(),
            'ultimateoptimizer-move-to-footer-js' => array(),
            'ultimateoptimizer-defer-js' => array()
        );
        $advanced= array_merge($default_advanced,$advanced);

        //CDN default
        $cdn = get_option('ultimateoptimizer_cdn_integration');
        if(empty($cdn)) $cdn = array();
        $wp_content = substr(WP_CONTENT_DIR,strlen(ABSPATH));
        $default_cdn = array(
            'cdn-active' => '0',
            'cdn-url' =>'',
            'cdn-content' => array('wp-includes',$wp_content),
            'cdn-exclude-content' => array('.php'),
            'cdn-relative-path' =>'1',
        );
        $cdn= array_merge($default_cdn,$cdn);

        // Varnish default
        $varnish = get_option('ultimateoptimizer_varnish_cache');
        if(empty($varnish)) $varnish = array();
        $default_varnish = array(
            'auto-purge-varnish' => '1',
        );
        $varnish= array_merge($default_varnish,$varnish);

        if(is_multisite()){
            $blogs = get_sites();
            foreach ($blogs as $blog){
                update_blog_option((int)$blog->blog_id,'ultimateoptimizer_basic_settings', $basic);
                 update_blog_option((int)$blog->blog_id,'ultimateoptimizer_must_settings', $must);
                update_blog_option((int)$blog->blog_id,'ultimateoptimizer_advanced_settings', $advanced);
                update_blog_option((int)$blog->blog_id,'ultimateoptimizer_cdn_integration', $cdn);
                update_blog_option((int)$blog->blog_id,'ultimateoptimizer_varnish_cache', $varnish);
            }
        }else{
            update_option('ultimateoptimizer_basic_settings', $basic);
            update_option('ultimateoptimizer_must_settings', $must);
            update_option('ultimateoptimizer_advanced_settings', $advanced);
            update_option('ultimateoptimizer_cdn_integration', $cdn);
            update_option('ultimateoptimizer_varnish_cache', $varnish);
        }

        //add header to htaccess if setting is enabled or by default if first installed
	    if ($basic['ultimateoptimizer-browser-cache'] == 1) {
		    Ultimateoptimizer_Configuration::add_expires_header( true );
	    }
	    if ($basic['ultimateoptimizer-gzip-compression'] == 1) {
		    Ultimateoptimizer_Configuration::add_gzip_htacess( true );
	    }
        //automatic config start cache
        Ultimateoptimizer_ConfigCache::factory()->write();
        Ultimateoptimizer_ConfigCache::factory()->write_config_cache();

        if ( !empty($basic) && !empty($basic['ultimateoptimizer-active'] )) {
            Ultimateoptimizer_ConfigCache::factory()->toggle_caching( true );
        }
    }

    /*
     * Register deactive plugin hook
     */
    public static function plugin_deactive_hook(){
        WP_Filesystem();
        Ultimateoptimizer_ConfigCache::factory()->clean_up();
        Ultimateoptimizer_ConfigCache::factory()->clean_config();
        Ultimateoptimizer_ConfigCache::factory()->toggle_caching(false);
	    Ultimateoptimizer_Configuration::add_expires_header(false);
	    Ultimateoptimizer_Configuration::add_gzip_htacess(false);
    }

    /*
     * Render tab
     */
    public static function render($tab){
        require_once (ULTIMATEOPTIMIZER_PLUGIN_DIR . 'views/tabs/'.$tab.'.php');
    }

    // Check varnish cache exist
    public static function check_varnish(){
        if(isset($_SERVER['HTTP_X_VARNISH'])){
            return true;
        }
        return false;
    }

    // Applied to the list of links to display on the plugins page
    public function ultimateoptimizer_add_action_links($links){
	    $mylinks = array(
		    '<a href="' . admin_url( 'options-general.php?page=ultimateoptimizer' ) . '">Settings</a>',
	    );

        return array_merge($mylinks, $links);
    }

	// Applied to the list of links to display on the network plugins page
	public function ultimateoptimizer_add_action_links_network($links){
		$mylinks = array(
			'<a href="' . network_admin_url( 'settings.php?page=ultimateoptimizer' ) . '">Settings</a>',
		);

		return array_merge($mylinks, $links);
	}

	// Clear all cache action
	public function ultimateoptimizer_clear_all_cache() {
		//delete minify
		Ultimateoptimizer_MinificationCache::clear_minification();
		//clear normal cache
		Ultimateoptimizer_PurgeCache::ultimateoptimizer_cache_flush();
		//clear varnish cache
		$this->ultimateoptimizer_clear_varnish();
	}

	// Clear all varnish cache action
	public function ultimateoptimizer_clear_varnish() {
		$main = new Ultimateoptimizer_PurgeVarnish();

		if (is_multisite()) {
			$sites = get_sites();
			foreach ($sites as $site) {
				switch_to_blog($site->blog_id);
				$homepage = home_url().'/?ultimateoptimizer';
				$main->purge_cache($homepage);
			}
			restore_current_blog();
		} else {
			$homepage = home_url() . '/?ultimateoptimizer';
			$main->purge_cache( $homepage );
		}
	}
}

$admin = new Ultimateoptimizer_Admin();
